﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Shared.UserModels;
    using Hims.Shared.UserModels.ChargeModule;
    using Hims.Shared.UserModels.Laboratory;
    using Hims.Shared.UserModels.Scan.ScanTest;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Microsoft.CodeAnalysis.CSharp;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels.Filters;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/scan-test")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ScanTestController : BaseController
    {
        /// <summary>
        /// The coupon services.
        /// </summary>
        private readonly IScanTestService scanTestService;

        /// <summary>
        /// The charge module service.
        /// </summary>
        private readonly IChargeModuleService chargeModuleService;

        /// <summary>
        /// The scan service.
        /// </summary>
        private readonly IScanLogService scanLogService;

        /// <inheritdoc />
        public ScanTestController(IScanTestService scanTestService, IChargeModuleService chargeModuleService, IScanLogService scanLogService)
        {
            this.scanTestService = scanTestService;
            this.chargeModuleService = chargeModuleService;
            this.scanLogService = scanLogService;
        }

        /// <summary>
        /// The fetch labs async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch-details")]
        public async Task<ActionResult> FetchScanTestsAsync([FromBody] ScanTestMasterModel model, [FromHeader] LocationHeader header)
        {
            var models = model ?? new ScanTestMasterModel();
            model.LocationId = int.Parse(header.LocationId);
            var response = await this.scanTestService.FetchAllScanTests(models);
            return this.Success(response);
        }

        /// <summary>
        /// The fetch labs async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch-all")]
        public async Task<ActionResult> FetchAll([FromBody] ScanTestMasterModel model, [FromHeader] LocationHeader header)
        {
            var models = model ?? new ScanTestMasterModel();
            model.LocationId = int.Parse(header.LocationId);
            var response = await this.scanTestService.FetchAllAsync(models);
            return this.Success(response);
        }

        /// <summary>
        /// The fetch labs async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> FetchAltScanTestsAsync([FromBody] ScanTestMasterModel model, [FromHeader] LocationHeader header)
        {
            var models = model ?? new ScanTestMasterModel();
            model.LocationId = int.Parse(header.LocationId);
            var response = await this.scanTestService.FetchAltScanTests(models);
            return this.Success(response);
        }

        /// <summary>
        /// The fetch labs async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch-import-test")]
        public async Task<ActionResult> FetchImportTestsAsync([FromBody] ScanTestMasterModel model, [FromHeader] LocationHeader header)
        {
            var models = model ?? new ScanTestMasterModel();
            model.LocationId = int.Parse(header.LocationId);
            var response = await this.scanTestService.FetchImportTestsAsync(models);
            return this.Success(response);
        }


        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add-test")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] ScanTestInsertModel model, [FromHeader] LocationHeader header)
        {
            model = (ScanTestInsertModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var response = await this.scanTestService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Scan test has already been exists with us.");
                case -2:
                    return this.Conflict("Unable to find ModuleMaster Id, Please Try Later");
                case 0:
                    return this.ServerError();
            }

            if (response > 0)
            {
                var scanLogModel = new ScanLogModel
                {
                    AccountId = model.CreatedBy,
                    ScanLogTypeId = (int)ScanLogTypes.ScanTestMaster,
                    LogFrom = (short)model.LoginRoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LocationId = Convert.ToInt32(model.LocationId),
                    LogDescription = $" {model.ModifiedByName} has Added Scan Test : <b>{model.ScanTestName}</b> Successfully.",
                };
                await this.scanLogService.LogAsync(scanLogModel);
            }

            return this.Success("Scan Test has been added successfully.");
        }

        /// <summary>
        /// The update coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon updated successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update-test")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] ScanTestInsertModel model, [FromHeader] LocationHeader location)
        {
            model = (ScanTestInsertModel)EmptyFilter.Handler(model);
            model.LocationId = int.Parse(location.LocationId);
            var response = await this.scanTestService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given scan test has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            if (response > 0)
            {
                var scanLogModel = new ScanLogModel
                {
                    AccountId = model.ModifiedBy,
                    ScanLogTypeId = (int)ScanLogTypes.ScanTestMaster,
                    LogFrom = (short)model.LoginRoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LocationId = Convert.ToInt32(model.LocationId),
                    LogDescription = $" {model.ModifiedByName} has Updated Scan Test : <b>{model.ScanTestName}</b> Successfully.",
                };
                await this.scanLogService.LogAsync(scanLogModel);
            }
            return this.Success("Scan test has been updated successfully.");
        }

        /// <summary>
        /// The delete specialization.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Specialization deleted successfully.
        /// - 409 - Specialization can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("modify-test-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] ScanTestMasterModel model, [FromHeader] LocationHeader header)
        {
            model = (ScanTestMasterModel)EmptyFilter.Handler(model);
            var response = await this.scanTestService.ActivateOrDeactivateTest(model);
            var scanLogModel = new ScanLogModel
            {
                AccountId = model.ModifiedBy,
                ScanLogTypeId = (int)ScanLogTypes.ScanTestMaster,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LocationId = Convert.ToInt32(header.LocationId),
                LogDescription = response > 0 ? $" {model.ModifiedByName}</b> has {(model.Active == true ? "Activated" : "Deactivated")} Scan Test : <b>{model.ScanTestName}</b> successfully." : $" {model.ModifiedByName}</b> has tried to Deactivate the Scan Test : <b>{model.ScanTestName}</b> - Cannot Deactivate this Test as it is linked to a Machine.",
            };
            await this.scanLogService.LogAsync(scanLogModel);

            return this.Success(response);
        }

        /// <summary>
        /// The modify labs async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("modify-import-test")]
        public async Task<ActionResult> ModifyImportTestsAsync([FromBody] TestImportModel model, [FromHeader] LocationHeader header)
        {
            model = (TestImportModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var response = await this.scanTestService.ModifyImportTestAsync(model);
            try
            {
                if (response > 0)
                {
                    var scanLogModel = new ScanLogModel
                    {
                        AccountId = model.CreatedBy,
                        ScanLogTypeId = (int)ScanLogTypes.ScanTestMaster,
                        LogFrom = (short)model.LoginRoleId,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LocationId = Convert.ToInt32(model.LocationId),
                        LogDescription = $" {model.ModifiedByName} has Added Scan Test : <b>{model.ScanTestName}</b> Successfully.",
                    };
                    await this.scanLogService.LogAsync(scanLogModel);
                }
            }
            catch (Exception)
            {
                // ignore
            }
            return this.Success(response);
        }

        /// <summary>
        /// Fetches all scan tests asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-all-scan-test")]
        public async Task<ActionResult> FetchAllScanTestsAsync([FromBody] ScanTestMasterModel model)
        {
            model ??= new ScanTestMasterModel();
            var response = await this.scanTestService.FetchAllScanTestsAsync(model);
            return this.Success(response);
        }

        [HttpPost]
        [Route("fetch-test-codes")]
        public async Task<ActionResult> FetchAllScanTestCodesAsync([FromBody] ScanTestMasterModel model, [FromHeader] LocationHeader header)
        {
            var models = model ?? new ScanTestMasterModel();
            model.LocationId = int.Parse(header.LocationId);
            var response = await this.scanTestService.FetchAllScanTestCodesAsync(models);
            return this.Success(response);
        }


        /// <summary>
        /// Fetches the lab with charge asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-scan-with-charges")]
        public async Task<ActionResult> FetchScanWithChargeAsync([FromBody] ScanTestMasterServiceModel model, [FromHeader] LocationHeader location)
        {
            model ??= new ScanTestMasterServiceModel();
            model.LocationId = Convert.ToInt32(location.LocationId);
            var response = await this.scanTestService.FetchAllScanAsync(model);

            var getModulesData = await this.chargeModuleService.GetRequiredDetailForChargeFetchAsync("scan", (int)model.LocationId);
            if (model.ChargeCategoryId != null && model.ChargeCategoryId > 0)
            {
                getModulesData.ChargeCategoryId = model.ChargeCategoryId;
            }
            if (getModulesData == null)
            {
                return this.Success(new List<ScanTestMasterModel>());
            }

            foreach (var scan in response)
            {
                scan.Charges = new List<ChargeModuleDetailsModel>();
                getModulesData.ReferenceId = scan.ScanTestMasterId;
                scan.Charges = (await this.chargeModuleService.FetchGivenChargesAsync(getModulesData)).ToList();
            }

            return this.Success(response);
        }
    }
}